/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	pgpInactivity.c - keyboard/mouse inactivity detection
	

	$Id: pgpInactivity.c,v 1.1 1999/09/14 19:10:54 cpeterson Exp $
____________________________________________________________________________*/

#include "pgpUDx.h"
#include "PGPsdkDriver.h"
#include "pgpInactivity.h"
#include "pgpMisc.h"

//______________________________________________________
//
//	decrement the second timer and purge timed-out entries

static VOID
sInactivityTick (
	IN PPGPINACTIVITY				ppi,
	IN PGPInactivityCallbackFunc*	pcallback,
	IN PULONG						puserval)
{
	PPGPINACTIVITYTIMER*		pppit		= &(ppi->ptimerFirst);
	PPGPINACTIVITYTIMER			ppitThis	= NULL;

	while (*pppit)
	{
		ppitThis = *pppit;

		// decrement the counter
		--(ppitThis->lSecondsRemaining);

		// if timed-out, remove from list
		if ((ppitThis->lSecondsRemaining < 0) &&
			(ppitThis->callback))
		{
			// return the registered callback function
			*pcallback	= ppitThis->callback;
			*puserval	= ppitThis->ulUserValue;

			// repeating event?
			if (ppitThis->ulFlags & kPGPUDFlag_RepeatingInactivityTimer)
			{
				ppitThis->lSecondsRemaining = ppitThis->lTimeoutSeconds;
				pppit = &(ppitThis->ptimerNext);
			}
			// not a repeating event so mark this entry as unused
			else
			{
				ppitThis->callback = NULL;
			}

			return;
		}
		// otherwise just step to next entry in list
		else
			pppit = &(ppitThis->ptimerNext);
	}
}


//______________________________________________________
//
//	remove any unused entries from the list

static VOID
sCleanupList (
	IN PPGPINACTIVITY				ppi)
{
	PPGPINACTIVITYTIMER*		pppit		= &(ppi->ptimerFirst);
	PPGPINACTIVITYTIMER			ppitThis	= NULL;

	while (*pppit)
	{
		ppitThis = *pppit;

		// if unused, remove from list
		if (ppitThis->callback == NULL)
		{
			*pppit = ppitThis->ptimerNext;
			pgpDriverSecureFree (ppitThis);
		}

		// otherwise just step to next entry in list
		else
			pppit = &(ppitThis->ptimerNext);
	}
}


//______________________________________________________
//
//	create a new inactivity timer; pass back handle of timer

static VOID
sCreateInactivityTimer (
	IN PPGPINACTIVITY		ppi,
	IN PPGPINACTIVITYSTRUCT	ppis)
{
	PPGPINACTIVITYTIMER	ppitNew		= NULL;

	// insert item into list
	ppitNew = pgpDriverSecureAlloc (sizeof(PGPINACTIVITYTIMER));
	if (ppitNew)
	{
		// fill in data
		ppitNew->callback			= ppis->callback;
		ppitNew->ulUserValue		= ppis->ulUserValue;
		ppitNew->lTimeoutSeconds	= ppis->ulSeconds;
		ppitNew->ulFlags			= ppis->ulFlags;

		// set timeout
		ppitNew->lSecondsRemaining = ppitNew->lTimeoutSeconds;

		// add to front of list
		ppitNew->ptimerNext = ppi->ptimerFirst;
		ppi->ptimerFirst = ppitNew;

		// return info
		ppis->ulTimerHandle = (ULONG)ppitNew;
		ppis->ulError = kPGPUDError_NoErr;
	}
	else
		ppis->ulError = kPGPUDError_MemAllocError;
}


//______________________________________________________
//
//	destroy an existing inactivity timer based on handle

static VOID
sDestroyInactivityTimer (
	IN PPGPINACTIVITY		ppi,
	IN PPGPINACTIVITYSTRUCT	ppis)
{
	PPGPINACTIVITYTIMER*		pppit		= &(ppi->ptimerFirst);
	PPGPINACTIVITYTIMER			ppitThis	= NULL;

	while (*pppit)
	{
		ppitThis = *pppit;

		// is this the timer we're looking for?
		if (ppis->ulTimerHandle == (ULONG)ppitThis)
		{
			*pppit = ppitThis->ptimerNext;
			pgpDriverSecureFree (ppitThis);
			ppis->ulError = kPGPUDError_NoErr;
			return;
		}
		// otherwise just step to next entry in list
		else
			pppit = &(ppitThis->ptimerNext);
	}

	ppis->ulError = kPGPUDError_ItemNotFound;
}


//______________________________________________________
//
//	activity has occured, reset all counters

VOID
pgpProcessActivityEvent (
	IN PPGPINACTIVITY		ppi,
	IN PVOID				pCriticalSection)
{
	PPGPINACTIVITYTIMER		ppit;

	pgpDriverEnterCriticalSection (pCriticalSection, SPINLOCK);
	ppit = ppi->ptimerFirst;

	// walk list and reset all counters
	while (ppit)
	{
		ppit->lSecondsRemaining = ppit->lTimeoutSeconds;
		ppit = ppit->ptimerNext;
	}
	pgpDriverLeaveCriticalSection (pCriticalSection, SPINLOCK);
}


//______________________________________________________
//
//	process a timer event -- purge expired cache entries

VOID
pgpProcessInactivityTimerEvent (
	IN PPGPINACTIVITY		ppi,
	IN PVOID				pCriticalSection)
{
	PGPInactivityCallbackFunc	callback	= NULL;
	ULONG						userval;

	pgpDriverEnterCriticalSection (pCriticalSection, SPINLOCK);
	sInactivityTick (ppi, &callback, &userval);
	pgpDriverLeaveCriticalSection (pCriticalSection, SPINLOCK);

	if (callback)
	{
		(*callback)(userval);
	}
}


//______________________________________________________
//
//	Initialize the inactivity system

VOID
pgpInitInactivity (
	IN PPGPINACTIVITY		ppi)
{
	ppi->ptimerFirst = NULL;
}


//______________________________________________________
//
//	This is the function that processes the DeviceIoControl
//	call which is coming from an application.

VOID
pgpInactivityProcessOperation (
	IN PPGPINACTIVITY			ppi,
	IN PPGPINACTIVITYSTRUCT		ppis,
	IN PVOID					pCriticalSection)
{
	switch (ppis->ulOperation) {
	case kPGPUDOperation_CreateInactivityTimer :
		pgpDriverEnterCriticalSection (pCriticalSection, MUTEX);
		sCleanupList (ppi);
		sCreateInactivityTimer (ppi, ppis);
		pgpDriverLeaveCriticalSection (pCriticalSection, MUTEX);
		break;

	case kPGPUDOperation_DestroyInactivityTimer :
		pgpDriverEnterCriticalSection (pCriticalSection, MUTEX);
		sCleanupList (ppi);
		sDestroyInactivityTimer (ppi, ppis);
		pgpDriverLeaveCriticalSection (pCriticalSection, MUTEX);
		break;

	default :
		ppis->ulError = kPGPUDError_UndefinedOperation;
		break;
	}
}

